/*----------------------------------------------------------------------------
 * Simple Traffic Calculator - Calculates stuff about traffic.
 * Copyright (C) 2004-2005 Arne Bochem <stc@arneb.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *----------------------------------------------------------------------------
 * Version: 1.14
 *----------------------------------------------------------------------------
 * Compile with:
 *   gcc simple.c -o stc -O3 -lm
 * Or:
 *   make
 * Or, to get better years:
 *   make ct
 *----------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <errno.h>

/* Type for huge scalar values. */
typedef unsigned long long int scalar_t;

/* Be 1337 and use a macro to round stuff. */
#define ROUND(x) ((scalar_t)floor(x + 0.5))

/* We associate a unit with a multiplier. */
struct unit_map
{
	char unit;
	scalar_t multiplier;
};

/* Here we'll do the actual map processing. */
scalar_t parseUnit (struct unit_map *u_map, short units, char *input)
{
	double user_input;
	char *last_character;
	char unit;
	short iterator;
	short has_unit = 0;

	/* Set errno to zero. */
	errno = 0;

	/* Convert string to integer. */
	user_input = strtod(input, &last_character);

	/* Check for huge values. */
	if (errno == ERANGE)
	{
		perror("Huge number in input");
		abort();
	}

	/* Check for zero. */
	if (user_input == 0)
	{
		if (input == last_character)
		{
			perror("Can't convert input");
			abort();
		}
		else
		{
			fprintf(stderr, "Input must not be zero.\n");
		}
		exit(1);
	}

	/* Determine unit. */
	unit = last_character[0];

	/* Apply multiplier. */
	for (iterator = 0; iterator < units; iterator++)
	{
		/* Is this the unit the user wants? */
		if (unit == u_map[iterator].unit)
		{
			/* Do the multiplication and be done with it. */
			user_input *= u_map[iterator].multiplier;
			has_unit = 1;

			/* Beware of long units. */
			if (last_character[1] != 0)
			{
				fprintf(stderr, "Warning: Bad unit \"%s\". Assuming \"%c\".\n", last_character, unit);
			}

			break;
		}
	}

	/* Check for integer overflow. */
	if (user_input >= pow(2, sizeof(scalar_t) * 8))
	{
		fprintf(stderr, "Integer overflow.\n");
		exit(1);
	}

	/* Check for bad units and warn, if found. */
	if (!has_unit && (unit != 0))
	{
		fprintf(stderr, "Warning: Bad unit \"%c\". Assuming default unit.\n", unit);
	}

	/* Done. Return rounded value. */
	return(ROUND(user_input));
}

scalar_t parseTime (char *input)
{
	/* Set up unit map. */
	struct unit_map u_map[6] =
	         {{'m', 60},                       /* Minute: 60s */
	          {'h', 60 * 60},                  /* Hour: 60m   */
	          {'d', 60 * 60 * 24},             /* Day: 24h    */
	          {'w', 60 * 60 * 24 * 7},         /* Week: 7d    */
	          {'M', 60 * 60 * 24 * 30},        /* Month: 30d  */
	                                           /* The year:   */
#ifndef CORRECT_TIME
	          {'y', 60 * 60 * 24 * 365}};      /* 365d        */
#else
	          {'y', 60 * 60 * 24 * 365.2425}}; /* 365.2425d   */
#endif

	/* Call unit map processor and get it done. */
	return(parseUnit(u_map, 6, input));
}

scalar_t parseSize (char *input)
{
	/* Set up unit map. */
	struct unit_map u_map[3] = {{'K', 1024},                /* KB: 1024B  */
	                            {'M', 1024 * 1024},         /* MB: 1024KB */
	                            {'G', 1024 * 1024 * 1024}}; /* GB: 1024MB */

	/* Call unit map processor and get it done. */
	return(parseUnit(u_map, 3, input));
}

scalar_t parseBandwidth (char *input)
{
	/* Set up unit map. Please note: 1B = 8b. */
	struct unit_map u_map[6] = {{'k', 1000 / 8},               /* Kb: 1000b  */
	                            {'m', 1000 * 1000 / 8},        /* Mb: 1000Kb */
	                            {'g', 1000 * 1000 * 1000 / 8}, /* Gb: 1000Mb */
	                            {'K', 1024},                   /* KB: 1024B  */
	                            {'M', 1024 * 1024},            /* MB: 1024KB */
	                            {'G', 1024 * 1024 * 1024}};    /* GB: 1024GB */

	/* Call unit map processor and get it done. */
	return(parseUnit(u_map, 6, input));
}

struct transform_map
{
	char *name;
	scalar_t divisor;
};

void humanUnit (struct transform_map *t_map, short transformations, char *default_unit, scalar_t value)
{
	scalar_t number;
	short iterator;
	short separator = 0;

	if (!value)
	{
		printf("0\n");
		return;
	}

	/* Process each unit. */
	for (iterator = 0; iterator < transformations; iterator++)
	{
		if (value >= t_map[iterator].divisor)
		{
			/* Calculate number of [unit]s and remainder. */
			number = value / t_map[iterator].divisor;
			value = value % t_map[iterator].divisor;

			/* Check if we need a unit separator. If we do, which one? */
			if (separator)
				printf("%s ", ((value) ? "," : " and"));

			/* Print the number of [unit]s and the unit, with an s, if it's plural. */
			printf("%Lu %s%s", number, t_map[iterator].name, ((number > 1) ? "s" : ""));

			/* We did print something and will be needing a separator from now on. */
			separator = 1;
		}
	}

	/* Now we have to check if there is anything left to print. */
	if (value)
	{
		if (separator)
			printf(" and ");
		printf("%Lu %s%s", value, default_unit, ((value > 1) ? "s" : ""));
	}

	/* Done. */
	printf("\n");
}

void humanTime (scalar_t time)
{
	/* Set up transform map. */
#ifndef CORRECT_TIME
	struct transform_map t_map[6] = {{"year", 60 * 60 * 24 * 365},
#else
	struct transform_map t_map[6] = {{"year", 60 * 60 * 24 * 365.2425},
#endif
	                                 {"month", 60 * 60 * 24 * 30},
	                                 {"week", 60 * 60 * 24 * 7},
	                                 {"day", 60 * 60 * 24},
	                                 {"hour", 60 * 60},
	                                 {"minute", 60}};

	/* Call transform map processor and get it done. */
	printf("Time: ");
	humanUnit(t_map, 6, "second", time);
}

void humanSize (scalar_t size)
{
	/* Set up transform map. Kibi? Mebi? Gibi? Yeah. We comply to standards... */
	struct transform_map t_map[3] = {{"gibibyte", 1024 * 1024 * 1024},
	                                 {"mebibyte", 1024 * 1024},
	                                 {"kibibyte", 1024}};

	/* Call transform map processor and get it done. */
	printf("Size: ");
	humanUnit(t_map, 3, "byte", size);
}

void humanBandwidth (scalar_t bandwidth)
{
	/* Set up transform map. */
	struct transform_map t_map[3] = {{"gibibyte", 1024 * 1024 * 1024},
	                                 {"mebibyte", 1024 * 1024},
	                                 {"kibibyte", 1024}};

	/* Call transform map processor and get it done. */
	printf("Transferred per second: ");
	humanUnit(t_map, 3, "byte", bandwidth);
}

void outputResult (scalar_t size, scalar_t time, scalar_t bandwidth, short human_readable)
{
	/* Check what kind of output is needed. */
	if (!human_readable)
	{
		/* Just print the values. Easy. */
		printf("%Lu %Lu %Lu\n", time, size, bandwidth);
	}
	else
	{
		/* Here we go. Convert-o-matic. */
		humanTime(time);
		humanSize(size);
		humanBandwidth(bandwidth);
	}
}

void getTime (scalar_t *size, scalar_t *time, scalar_t *bandwidth)
{
	/* Calculate time. */
	*time = *size / *bandwidth;
}

void getSize (scalar_t *size, scalar_t *time, scalar_t *bandwidth)
{
	/* Calculate time. */
	*size = *time * *bandwidth;
}

void getBandwidth (scalar_t *size, scalar_t *time, scalar_t *bandwidth)
{
	/* Calculate time. */
	*bandwidth = *size / *time;
}

void printShortHelp (scalar_t *size, scalar_t *time, scalar_t *bandwidth)
{
	printf("Usage: stc MODE VALUE1 VALUE2 [OPTION]\n\n\
  --get-time         -gt Get time from size and bandwidth.\n\
  --get-size         -gs Get size from time and bandwidth.\n\
  --get-bandwidth    -gb Get bandwidth from time and size.\n\
  --help             -h  Print this usage information.\n\
  --version          -V  Print version.\n\
  --size             -s Size [Default: byte. Units: K, M, G]\n\
  --time             -t Time [Default: seconds. Units: m, h, d, w, M, y]\n\
  --bandwidth        -b Bandwidth [Default: bytes per second (K = 1024). Units:\n\
                        K, M, G. Additional Units: k, m, g (bits per second)\n\
                        (k = 1000)]\n\
  --machine-readable -m Print in machine readable format. (Format:\n\
                        TIME SIZE BANDWIDTH)\n\n"); 
}
void printHelp (scalar_t *size, scalar_t *time, scalar_t *bandwidth)
{
	printf("Usage: stc MODE VALUE1 VALUE2 [OPTION]\n\n\
Simple Traffic Calculator 1.14 - Calculates stuff about traffic.\n\
Copyright (C) 2004-2005 Arne Bochem <stc@arneb.de>\n\n\
Modes:\n\
  --get-time         -gt Get time from size and bandwidth.\n\
  --get-size         -gs Get size from time and bandwidth.\n\
  --get-bandwidth    -gb Get bandwidth from time and size.\n\
  --help             -h  Print this usage information.\n\
  --version          -V  Print version.\n\n\
Values:\n\
  --size             -s Size [Default: byte. Units: K, M, G]\n\
  --time             -t Time [Default: seconds. Units: m, h, d, w, M, y]\n\
  --bandwidth        -b Bandwidth [Default: bytes per second (K = 1024). Units:\n\
                        K, M, G. Additional Units: k, m, g (bits per second)\n\
                        (k = 1000)]\n\n\
Options:\n\
  --machine-readable -m Print in machine readable format. (Format:\n\
                        TIME SIZE BANDWIDTH)\n\n\
Notes:\n\
  Machine-readable:\n\
    Size is in bytes.\n\
    Time is in seconds.\n\
    Bandwidth is in bytes per second.\n\
  Time units:\n\
    A month is 30 days.\n\
    A year is %s days.\n\
  Beware of huge numbers. There are no overflow checks yet. Please\n\
  sanity-check data.\n\n",
#ifndef CORRECT_TIME
		"365"
#else
		"365.2425"
#endif
	); 
}

void printVersion (scalar_t *size, scalar_t *time, scalar_t *bandwidth)
{
	printf("1.14\n");
}

int main (int argc, char *argv[])
{
	/* Initialize function map. */
	void (*(f_map[6])) (scalar_t *, scalar_t *, scalar_t *) = {getTime, getSize, getBandwidth, printHelp, printVersion, printShortHelp};

	/* Initialize argument names for function map. */
	char *fp_map[5][2] = {{"--get-time", "-gt"}, {"--get-size", "-gs"}, {"--get-bandwidth", "-gb"}, {"--help", "-h"}, {"--version", "-V"}};

	/* Initialize default values. */
	scalar_t size = 0;          /* This be bytes.             */
	scalar_t time = 0;          /* This be seconds.           */
	scalar_t bandwidth = 0;     /* This be bytes per second.  */
	short human_readable = 1;   /* Print pretty lines.        */
	short function = -1;        /* Nothing found yet.         */
	short fp_iterator;          /* We'll need this too.       */
	short fun_def;              /* We found something.        */
	int iterator;               /* We'll need this, you know. */

	/* Get arguments. */
	for (iterator = 1; iterator < argc; iterator++)
	{
		/* Look for function in the argument. */
		fun_def = 0;
		for (fp_iterator = 0; fp_iterator < 5; fp_iterator++)
		{
			if (!strcmp(argv[iterator], fp_map[fp_iterator][0]) || !strcmp(argv[iterator], fp_map[fp_iterator][1]))
			{
				if (function != -1)
					fprintf(stderr, "Warning: Changing mode from %s to %s.\n", fp_map[function][0] + 2, fp_map[fp_iterator][0] + 2);
				function = fp_iterator;
				fun_def = 1;
				break;
			}
		}

		/* Found one. No need to look further. */
		if (fun_def)
			continue;

		/* Look for other kinds of arguments. */
		if (!strcmp(argv[iterator], "--machine-readable") || !strcmp(argv[iterator], "-m"))
		{
			human_readable = 0;
			continue;
		}
		if (!strcmp(argv[iterator], "--size") || !strcmp(argv[iterator], "-s"))
		{
			/* Ok, let's look at the next argument. */
			iterator++;
			size = parseSize(argv[iterator]);
			continue;
		}
		if (!strcmp(argv[iterator], "--time") || !strcmp(argv[iterator], "-t"))
		{
			/* Ok, let's look at the next argument. */
			iterator++;
			time = parseTime(argv[iterator]);
			continue;
		}
		if (!strcmp(argv[iterator], "--bandwidth") || !strcmp(argv[iterator], "-b"))
		{
			/* Ok, let's look at the next argument. */
			iterator++;
			bandwidth = parseBandwidth(argv[iterator]);
			continue;
		}

		/* Got invalid argument. Give error message. */
		fprintf(stderr, "Error: Invalid argument: %s\n", argv[iterator]);

		/* And try to help the user. */
		function = 5;

		/* But we failed anyway. */
		break;
	}

	/* Fallback function is printHelp. */
	if (function == -1)
		function = 3;

	/* Call function. */
	f_map[function](&size, &time, &bandwidth);

	/* Real function used? */
	if (function > 2)
	{
		/* No, just return. */
		return(1);
	}
	else
	{
		/* Yeah, output results and exit successfully. */
		outputResult(size, time, bandwidth, human_readable);
		return(0);
	}
}
